/************************************************************************
 * @file: AudioBackend.h
 *
 * @version: 0.1
 *
 * @description: This header file contains definition for Interface class
 * Backend. This class will be inherited by all the Backend [eg: Alsa] and
 * will implement the member functions. Based on the Backed library name
 * provide by Application to the Selector class, the function calls will
 * be delegated to appropriate Backend libraries.
 *
 * @authors: Jens Lorenz, jlorenz@de.adit-jv.com 2015
 *           Thouseef Ahamed, tahamed@de.adit-jv.com 2015
 *           Vijay Palaniswamy, vijay.palaniswamy@in.bosch.com 2015
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#ifndef _ADIT_UTILITY_AUDIOBACKEND_H_
#define _ADIT_UTILITY_AUDIOBACKEND_H_

#include <string>

#include "AudioTypes.h"

namespace adit
{

namespace utility
{

namespace audio
{

class Backend
{

public:
    Backend(void) {};
    virtual ~Backend(void) {};

    /**
     * This function will be called to open the stream for playback/capture with the specified parameters.
     * @param [IN]     in           Name for inward direction (Capture).
     * @param [IN]     out          Name for outward direction (Playback).
     * @param [IN]     format       Specify data format of stream is Little or Big endian, Signed or Unsigned, 16-bit or some other type.
     * @param [IN]     rate         The sample rate of the stream
     * @param [IN]     channels     The count of channels of stream
     * @param [IN|OUT] frames       Period size in frames
     * @return OK          if the stream is opened successfully
     * @return BUSY        if specified Device/Card/module-name is already opened/busy
     * @return UNSUPPORTED if not able to configure stream with specified parameters [Channels, Rate ...]
     * @return INVALID     if trying to start/stop/abort the stream, without opening
     * @return FAILURE     if specified streaming device is busy/invalid & specified parameters not matching
     * @note
     * Based on Backend the meaning for \c in or \c out varies,
     * @li If ALSA       in/out represents a virtual ALSA device or card
     * @li If PulseAudio in/out represents the module-name
     *
     * Configuring stream modes
     * @li In case of playback mode the in device name should be empty
     * @li In case of capture mode the out device name should be empty
     * @li In case of duplex both device name should be available
     *
     * frames[Period Size]:
     * Application can pass the specific Period Size (frames) value to backend, Configured value will be updated in frames field.
     * Configured value can be different from Application passed value [In case devices didn't support the passed value]
     * @li In case of playback mode, configured period frames data should be sent to backend using processing callback
     * @li In case of capture mode, configured period frames data will be sent to Application using processing callback
     */
    virtual AudioError openStream(const std::string &in, const std::string &out,
            const AudioFormat format, const uint32_t rate, const uint32_t channels,
            uint32_t &frames) = 0;

    /**
     * This function will be called to close the stream and frees any associated stream memory.
     * @return OK          if the stream is closed successfully
     * @return INVALID     if trying to close stream, without stopping
     * @return FAILURE     if specified streaming device closing fails
     */
    virtual AudioError closeStream(void) = 0;

    /**
     * This function will be called to start Playback/Capture of stream
     * @return OK          if the stream is started successfully
     * @return INVALID     if trying to start stream, without opening
     * @return FAILURE     if specified streaming device prepare/start fails
     * @note If Fade-In is configured for specific value, it will be processed only for
     *       capture stream till application configured time
     *       By default 8 milliseconds of Fade-In is configured to avoid click/pop noise
     *       This feature can be deactivated by setFadeTime(FadeMode::IN, StreamDirection::IN, 0)
     */
    virtual AudioError startStream(void) = 0;

    /**
     * This function will be called to stop Playback/Capture of stream
     * @return OK          if stream is stopped successfully
     * @return FAILURE     if specified streaming device stop failed
     * @return INVALID     if trying to stop the stream, without opening
     * @note During stop, remaining samples in output queue will be played.
     *       If Fade-Out is configured for specific value, it will be processed only for
     *       Playback stream till application configured time.
     *       Therefore application has to provide data till the configure Fade time,
     *       because callback will be called from Backend to get data from application
     *       till reaching the configured fade time.
     *       By default 8 milliseconds of Fade-Out is configured to avoid click/pop noise.
     *       This feature can be deactivated by setFadeTime(FadeMode::OUT, StreamDirection::OUT, 0)
     */
    virtual AudioError stopStream(void) = 0;

    /**
     * This function will be called to abort Playback/Capture of stream
     * @return OK          if stream is aborted successfully
     * @return FAILURE     if specified streaming device abort opening
     * @return INVALID     if trying to abort the stream, without opening
     * @note During abort Fade-In/Fade-Out will not be applied for capture/playback stream
     *       Also, remaining samples in input/output queue will be discarded
     */
    virtual AudioError abortStream(void) = 0;

    /**
     * This function will be called to get stream time in Frames
     * @return Returns Stream time in Frame count
     * @return 0 if stream is not started
     */
    virtual uint64_t getStreamTimeInFrames(void) = 0;

    /**
     * This function will be called to get stream Latency time in Frames
     * @return Returns Stream Latency in Frame count
     * @return 0 if stream not opened or started
     */
    virtual uint64_t getStreamLatencyInFrames(void) = 0;

    /**
     * This function will be called for configuring the Fade-In or Fade-Out Time
     * @param [IN] mode      Specify Fade Mode [Fade-In/Fade-Out]
     * @param [IN] direction Specify stream direction Playback/Capture
     * @param [IN] time      Specify the Fade-In/Fade-Out time to configure
     * @return OK          if fade time is configured successfully
     * @return UNSUPPORTED if the fader doesn't support the specified fade time [Fade time Range: 0 - 500 milliseconds]
     * @note
     * Currently supported Fading combinations
     * @li mode = FadeMode::IN  and direction = StreamDirection::IN
     * @li mode = FadeMode::OUT and direction = StreamDirection::OUT
     * Unsupported Fading combinations
     * @li mode = FadeMode::OUT and direction = StreamDirection::IN
     * @li mode = FadeMode::IN  and direction = StreamDirection::OUT
     */
    virtual AudioError setFadeTime(const enum FadeMode mode, const enum StreamDirection direction, const uint32_t time) = 0;

    /**
     * This function will be called for getting configured Fade-In or Fade-Out Time
     * @param [IN]  mode      Specify Fade Mode [Fade-In/Fade-Out]
     * @param [IN]  direction Specify stream direction Playback/Capture
     * @param [OUT] time      To get the configured Fade-In/Fade-Out time
     * @return OK          if configured fade time is returned successfully
     * @return UNSUPPORTED if fader was not configured
     */
    virtual AudioError getFadeTime(const enum FadeMode mode, const enum StreamDirection direction, uint32_t& time) = 0;

    /**
     * This function will be called to get the name of selected Backend
     * @return Name of the selected backed
     */
    virtual const char* getBackendName(void) = 0;

    /**
     * This function will be called to set the worker thread priority
     * @param [IN] policy   To specify thread policy for same priority Threads
     * @param [IN] priority Thread Priority value
     *
     * @note The input value for policy
     * @li SCHED_OTHER can only be used at static priority 0
     * @li SCHED_FIFO have a priority value in the range 1 (low) to 99 (high)
     * @li SCHED_RR have a priority value in the range 1 (low) to 99 (high)
     *
     * @note Default priority of worker thread is derived from process.
     * @li Thread policy and priority should be configured before start of stream.
     * @li Thread policy and priority can be changed after start call but this is not recommended.
     */
    virtual void setThreadSched(int policy, int priority) = 0;

    /**
     * This function will be called to set the initial time out for Playback/Capture
     * @param [IN] timeout   Specify the initial timeout value in milliseconds
     */
    virtual void setInitialTimeout(const uint32_t timeout) = 0;

    /**
     * This function will be called to get the initial time out in milliseconds
     * @return Returns the initial timeout value in milliseconds
     * @note Default initial timeout is 1000 ms
     */
    virtual uint32_t getInitialTimeout(void) const = 0;
};

} /* namespace audio */

} /* namespace utility */

} /* namespace adit */

#endif /* _ADIT_UTILITY_AUDIOBACKEND_H_ */
